package NICE;

use strict;
use warnings;

use Bit::Vector::Overload;
 Bit::Vector->Configuration("in=hex,ops=arithmetic,out=hex");

use Carp;

our $VERSION = 0.4;

## Export Settings
use base qw/Exporter/;

our @EXPORT_OK = qw/
   assemble
   disassemble
   init_code
   init_memory
   init_registers
   loadbin
   step
   @Status_Bits
/;

our %EXPORT_TAGS = ( all => [ @EXPORT_OK ] );

use constant INSTR_SIZE => 32;
use constant REGISTERS  => 16;

## Paramter Definitions

our @Status_Bits = qw(1 X C Z N V M I);

my %Instructions = (
   ALU => {
      TYPE => 0,
      INSTR => {
         MOVE => {
            OPCODE  => 0x0,
            OPCOUNT => 1,
            EXEC    => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               return;
            },
         },
         SUB  => {
            OPCODE  => 0x1,
            OPCOUNT => 2,
            EXEC    => sub {
               my($p, $use_carry, $c_in) = @_;
               return(
                  $p->[0]{value}->subtract(
                     $p->[1]{value},
                     $p->[2]{value},
                     $use_carry && $c_in,
                  )
               );
            },
         },
         MDBL  => {
            OPCODE  => 0x2,
            OPCOUNT => 2,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               return(
                  $p->[0]{value}->add(
                     $p->[1]{value},
                     $p->[1]{value} & $p->[2]{value},
                     $use_carry && $c_in,
                  )
               );
            },
         },
         ADD  => {
            OPCODE  => 0x3,
            OPCOUNT => 2,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               return(
                  $p->[0]{value}->add(
                     $p->[1]{value},
                     $p->[2]{value},
                     $use_carry && $c_in,
                  )
               );
            },
         },
         DBL  => {
            OPCODE  => 0x4,
            OPCOUNT => 1,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               return(
                  $p->[0]{value}->add(
                     $p->[1]{value},
                     $p->[1]{value},
                     $use_carry && $c_in,
                  )
               );
            },
         },
         DEC  => {
            OPCODE  => 0x5,
            OPCOUNT => 1,
            EXEC => sub {
               my($p) = @_;
               return(0, $p->[0]{value}->dec($p->[1]{value}));
            },
         },
         NOT  => {
            OPCODE  => 0x6,
            OPCOUNT => 1,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->Not($p->[1]{value});
               return;
            },
         },
         NOR  => {
            OPCODE  => 0x7,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->Or($p->[1]{value}, $p->[2]{value});
               $p->[0]{value}->Flip();
               return;
            },
         },
         IAND => {
            OPCODE  => 0x8,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[1]{value}->Flip();
               $p->[0]{value}->And($p->[1]{value}, $p->[2]{value});
               return;
            },
         },
         NAND => {
            OPCODE  => 0x9,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->And($p->[1]{value}, $p->[2]{value});
               $p->[0]{value}->Flip();
               return;
            },
         },
         XOR  => {
            OPCODE  => 0xA,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->Xor($p->[1]{value}, $p->[2]{value});
               return;
            },
         },
         IOR  => {
            OPCODE  => 0xB,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[1]{value}->Flip();
               $p->[0]{value}->Or($p->[1]{value}, $p->[2]{value});
               return;
            },
         },
         XNOR => {
            OPCODE  => 0xC,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->Xor($p->[1]{value}, $p->[2]{value});
               $p->[0]{value}->Flip();
               return;
            },
         },
         AND  => {
            OPCODE  => 0xD,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->And($p->[1]{value}, $p->[2]{value});
               return;
            },
         },
         ONE  => {
            OPCODE  => 0xE,
            OPCOUNT => 0,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = '1';
               return;
            },
         },
         OR   => {
            OPCODE  => 0xF,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value}->Or($p->[1]{value}, $p->[2]{value});
               return;
            },
         },
      },
   },
   SHIFT => {
      TYPE => 1,
      INSTR => {
         SHL => {
            OPCODE  => 0x0,
            OPCOUNT => 2,
            ALIAS   => 1,
         },
         ROL => {
            OPCODE  => 0x6,
            OPCOUNT => 2,
            ALIAS   => 1,
         },
         BPL => {
            OPCODE  => 0x7,
            OPCOUNT => 2,
            ALIAS   => 1,
         },
         SHR => {
            OPCODE  => 0x8,
            OPCOUNT => 2,
            ALIAS   => 1,
         },
         ROR => {
            OPCODE  => 0xE,
            OPCOUNT => 2,
            ALIAS   => 1,
         },
         BPR => {
            OPCODE  => 0xF,
            OPCOUNT => 2,
            ALIAS   => 1,
         },
         '0<0' => {
            OPCODE  => 0x0,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               $p->[0]{value}->Move_Left($p->[2]{value}->to_Dec());
               return;
            },
         },
         '0<1' => {
            OPCODE  => 0x1,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $bits = ($p->[2]{value} & 0x1F)->to_Dec();
               $p->[0]{value}->Move_Left($bits);
               my $mask = oct('0b' . ('1' x $bits));
               $p->[0]{value} |= $mask;
               return;
            },
         },
         '0<C' => {
            OPCODE  => 0x2,
            OPCOUNT => 2,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $bits = ($p->[2]{value} & 0x1F)->to_Dec();
               $p->[0]{value}->Move_Left($bits);
               return unless $c_in;
               my $mask = oct('0b' . ('1' x $bits));
               $p->[0]{value} |= $mask;
               return;
            },
         },
         'C<0' => {
            OPCODE  => 0x3,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $c_out;
               $c_out = $p->[0]{value}->shift_left(0)
                  for (1..$p->[2]{value} & 0x1F);
               return($c_out);
            },
         },
         'C<1' => {
            OPCODE  => 0x4,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $c_out;
               $c_out = $p->[0]{value}->shift_left(1)
                  for (1..$p->[2]{value} & 0x1F);
               return($c_out);
            },
         },
         'C<C' => {
            OPCODE  => 0x5,
            OPCOUNT => 2,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               $p->[0]{value} = $p->[1]{value};
               $c_in = $p->[0]{value}->shift_left($c_in)
                  for (1..$p->[2]{value} & 0x1F);
               return($c_in);
            },
         },
         '<<<' => {
            OPCODE  => 0x6,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               $p->[0]{value}->rotate_left()
                  for (1..$p->[2]{value} & 0x1F);
               return;
            },
         },
         'C<<' => {
            OPCODE  => 0x7,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $c_out;
               $c_out = $p->[0]{value}->rotate_left()
                  for (1..$p->[2]{value} & 0x1F);
               return($c_out);
            },
         },
         '0>0' => {
            OPCODE  => 0x8,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               $p->[0]{value}->Move_Right($p->[2]{value}->to_Dec());
               return;
            },
         },
         '1>0' => {
            OPCODE  => 0x9,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $bits = ($p->[2]{value} & 0x1F)->to_Dec();
               $p->[0]{value}->Move_Right($bits);
               my $mask = oct('0b' . ('1' x $bits));
               $p->[0]{value} |= $mask;
               return;
            },
         },
         'C>0' => {
            OPCODE  => 0xA,
            OPCOUNT => 2,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $bits = ($p->[2]{value} & 0x1F)->to_Dec();
               $p->[0]{value}->Move_Right($bits);
               return unless $c_in;
               my $mask = oct('0b' . ('1' x $bits));
               $p->[0]{value} |= $mask;
               return;
            },
         },
         '0>C' => {
            OPCODE  => 0xB,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $c_out;
               $c_out = $p->[0]{value}->shift_right(0)
                  for (1..$p->[2]{value} & 0x1F);
               return($c_out);
            },
         },
         '1>C' => {
            OPCODE  => 0xC,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $c_out;
               $c_out = $p->[0]{value}->shift_right(1)
                  for (1..$p->[2]{value} & 0x1F);
               return($c_out);
            },
         },
         'C>C' => {
            OPCODE  => 0xD,
            OPCOUNT => 2,
            EXEC => sub {
               my($p, $use_carry, $c_in) = @_;
               $p->[0]{value} = $p->[1]{value};
               $c_in = $p->[0]{value}->shift_right($c_in)
                  for (1..$p->[2]{value} & 0x1F);
               return($c_in);
            },
         },
         '>>>' => {
            OPCODE  => 0xE,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               $p->[0]{value}->rotate_right()
                  for (1..$p->[2]{value} & 0x1F);
               return;
            },
         },
         '>>C' => {
            OPCODE  => 0xF,
            OPCOUNT => 2,
            EXEC => sub {
               my($p) = @_;
               $p->[0]{value} = $p->[1]{value};
               my $c_out;
               $c_out = $p->[0]{value}->rotate_right()
                  for (1..$p->[2]{value} & 0x1F);
               return($c_out);
            },
         },
      },
   },
   HALT => {
      TYPE  => 2,
      INSTR => {
         'HALT' => {
            OPCODE  => 0x0,
            OPCOUNT => -1,
         }
      },
   },
);


my %Opcode_Names = map {
   my $type = $_;
   $type => {
      map {
         exists($Instructions{$type}{INSTR}{$_}{ALIAS}) ?
         () :
         ($Instructions{$type}{INSTR}{$_}{OPCODE} => $_);
      } keys(%{$Instructions{$type}{INSTR}})
   };
} keys(%Instructions);


my %Mnemonics = map {
   my $type = $_;
   $_ => [
      sort {
         $Instructions{$type}{INSTR}{$a}{OPCODE}
             <=>
         $Instructions{$type}{INSTR}{$b}{OPCODE};
      } grep {
         ! exists($Instructions{$type}{INSTR}{$_}{ALIAS})
      } keys(%{$Instructions{$type}{INSTR}})
   ];
} keys(%Instructions);


my %Instr_TypeNr = map {
   $Instructions{$_}{TYPE} => $_;
} keys(%Instructions);

my %Instr_Type = map {
   my $type = $_;
   map { $_ => $type } keys(%{$Instructions{$_}{INSTR}});
} keys(%Instructions);

my %Ops = map {
   my $type = $_;
   map {
      exists($Instructions{$type}{INSTR}{$_}{EXEC}) ?
      ($_ => $Instructions{$type}{INSTR}{$_}{EXEC}) :
      ()
   } keys(%{$Instructions{$_}{INSTR}});
} keys(%Instructions);


my %Valid_Dir = map { $_ => 1 } qw/.ASCII_B .ASCII_L .BLK .EQU .ORG .DAT/;

my %Valid_Inst = map {
   my $type = $_;
   map {
      $_ => $type
   } keys(%{$Instructions{$_}{INSTR}})
} keys(%Instructions);

my %SB_Nr = map { $Status_Bits[$_] => $_ } 0..$#Status_Bits;

my %Mods  = qw/mod M use_carry C/;

my $ws = '__WS__';


## SUBS

sub disassemble {
   my($mem, $code, $changed_mem) = @_;

   croak('usage: disassemble(<mem_aref>, <code_aref>, [<changed_mem_href>])')
      unless
         ref($mem)  eq 'ARRAY' and
         ref($code) eq 'ARRAY' and
         (!$changed_mem or ref($changed_mem) eq 'HASH');

   # $changed_mem undef --> check all
   my @check = $changed_mem                           ?
               sort { $a <=> $b } keys(%$changed_mem) :
               0..$#{$mem}                            ;

   my %new_code;
   my $last = 0;
   for my $addr ( @check ) {

      # go back to last instruction, skipping constants
      while (
         $code->[$addr]        and
         $code->[$addr] eq ' ' and
         $addr > 0
      ) {
         $addr--;
      }

      # dissassemble until synch (tmp code eq old code)
      my $sync;
      my %tmp;
      while ( !$sync and $addr < @$mem ) {

         %tmp = ();

         # already done while backtracking ?
         next if exists($new_code{$addr});

         # no code for NOPs
         if ( $mem->[$addr] == 0 ) {
            $tmp{$addr} = '';
            next;
         }

         # parsing
         my $pi = _parse_instr($mem->[$addr]);

         # build condition field
         my $cond = '';
         if ( $pi->{negate} or $pi->{cond} ) {
            $cond .= $pi->{negate} ? '?!' : '?';
            $cond .= $Status_Bits[$pi->{cond}];
         }

         # build operands
         my @ops;
         my $cnbr = 1;
         for my $i ( 0..$pi->{opcount} ) {

            # check for indirect address mode
            my $op = $pi->{params}[$i]{addr} & 4 ? '@' : '';

            SWITCH: {
               $_ = $pi->{params}[$i]{addr} & 3;

               my $reg = 'R' . $pi->{params}[$i]{reg};

               /0/ && do {
                  # normal register
                  $op .= $reg;
                  last SWITCH;
               };

               /1/ && do {
                  # decrement (post for dest, pre for source)
                  $op .= ($i == 0 && $op eq '') ? "$reg--" : "--$reg";
                  last SWITCH;
               };

               /2/ && do {
                  # post increment
                  $op .= "$reg++";
                  last SWITCH;
               };

               /3/ && do {
                  # handle constants
                  my $const_addr = $addr + $cnbr++;
                  $op .= '0x' . $mem->[$const_addr];
                  $op .= "[$reg]" if $reg ne 'R0';
                  # constants are coded with single space
                  $tmp{$const_addr}  = ' ';
                  last SWITCH;
               };
            }
            # save operand
            push(@ops, $op);
         }

         # join operands column aligned
         my $opstr   = join(' ', map { sprintf("%-16s", $_) } @ops);

         # construct modifiers
         my @mod = map { $pi->{$_} ? $Mods{$_} : () } ('mod', 'use_carry');
         my $mod = @mod && $pi->{op_name} ne 'HALT' ?
                   join('', '[', @mod, ']')       :
                   '';

         # combine statement
         $tmp{$addr} =
            sprintf("%-3s %-8s %s", $cond, $pi->{op_name} . $mod, $opstr);

      } continue {

         $sync = (
            !defined($tmp{$addr}) or
            $code->[$addr] eq $tmp{$addr}
         );

         # only save to %new_code hash if changed
         unless ( $sync ) {
            for ( keys(%tmp) ) {
               $code->[$_]   = $tmp{$_};
               $new_code{$_} = $tmp{$_};
            }
         }

         $addr++;
      }
   }

   return(\%new_code);
}

sub step {
   my($mem, $reg, $changed) = @_;

   croak('usage: step(<mem_aref>, <reg_aref>, <changed_mem_href>)')
      unless
         ref($mem)     eq 'ARRAY' and
         ref($reg)     eq 'ARRAY' and
         ref($changed) eq 'HASH';

   # keep reg[0] clean and reg[14][1] set
   $reg->[0]  &= "0";
   $reg->[14] |= "01000000";

   my $instruct = $mem->[$reg->[15]->to_Dec()];
   ++$reg->[15];

   my $pi = _parse_instr($instruct);

   # getting additional constants
   for my $p ( @{$pi->{params}} ) {
      if ( ($p->{addr} & 3) == 3 ) {
         $p->{mem} = $mem->[$reg->[15]->to_Dec()];
         ++$reg->[15];
      }
   }

   my $exec_instr = $pi->{negate} ^ $reg->[14]->bit_test($pi->{cond} + 24);

   return(1) unless $exec_instr;

   return if $pi->{op_name} eq 'HALT';

   for my $i ( 0..$#{$pi->{params}} ) {
      _read_op($mem, $reg, $pi->{params}[$i], $i);
   }

   my($c_out, $ovfl) = $Ops{$pi->{op_name}}->(
      $pi->{params},
      $pi->{use_carry},
      $reg->[14]->bit_test(29),
   );

   _write_op(
      $mem,
      $reg,
      $pi->{mod},
      $c_out,
      $ovfl,
      $pi->{params}[0],
      $changed,
   );

   # keep reg[0] clean and reg[14][1] set
   $reg->[0]  &= "0";
   $reg->[14] |= "01000000";

   return(1);
}


sub _read_op {
   my($mem, $reg, $p, $incdec) = @_;

   my $rval;

   SWITCH: {
      $_ = $p->{addr} & 3;

      /0/ && do {
         $rval = $reg->[$p->{reg}];
         last SWITCH;
      };

      /1/ && do {
         --$reg->[$p->{reg}] if $incdec;
         $rval = $reg->[$p->{reg}];
         last SWITCH;
      };

      /2/ && do {
         $rval = $reg->[$p->{reg}];
         $reg->[$p->{reg}]++ if $incdec;
         last SWITCH;
      };

      /3/ && do {
         $rval = $reg->[$p->{reg}] + $p->{mem};
         last SWITCH;
      };
   }

   $p->{value} = $p->{addr} & 4 ? $mem->[$rval->to_Dec()] : $rval;
}


sub _write_op {
   my($mem, $reg, $mod, $c_out, $ovfl, $p, $mem_changed) = @_;

   SWITCH: {
      $_ = $p->{addr};

      /0/ && do {
         $reg->[$p->{reg}]->Copy($p->{value});
         last SWITCH;
      };

      /1/ && do {
         $reg->[$p->{reg}]->Copy($p->{value});
         $reg->[$p->{reg}]--;
         last SWITCH;
      };

      /2/ && do {
         $reg->[$p->{reg}]->Copy($p->{value});
         $reg->[$p->{reg}]++;
         last SWITCH;
      };

      /3|7/ && do {
         my $addr = $reg->[$p->{reg}]->to_Dec() + $p->{mem}->to_Dec();
         unless ( $mem->[$addr] == $p->{value} ) {
            $mem->[$addr]->Copy($p->{value});
            $mem_changed->{$addr} = 1;
         }
         last SWITCH;
      };

      /4/ && do {
         my $addr = $reg->[$p->{reg}]->to_Dec();
         unless ( $mem->[$addr] == $p->{value} ) {
            $mem->[$addr]->Copy($p->{value});
            $mem_changed->{$addr} = 1;
         }
         last SWITCH;
      };

      /5/ && do {
         $reg->[$p->{reg}]--;
         my $addr = $reg->[$p->{reg}]->to_Dec();
         unless ( $mem->[$addr] == $p->{value} ) {
            $mem->[$addr]->Copy($p->{value});
            $mem_changed->{$addr} = 1;
         }
         last SWITCH;
      };

      /6/ && do {
         my $addr = $reg->[$p->{reg}]->to_Dec();
         unless ( $mem->[$addr] == $p->{value} ) {
            $mem->[$addr]->Copy($p->{value});
            $mem_changed->{$addr} = 1;
         }
         $reg->[$p->{reg}]++;
         last SWITCH;
      };
   }

   if ( $mod ) {
      $reg->[14] &= '01000000';

      $reg->[14]->Bit_On(25) if $p->{value} == 0xFFFFFFFF;
      $reg->[14]->Bit_On(26) if $c_out;
      $reg->[14]->Bit_On(27) if $p->{value} == 0;
      $reg->[14]->Bit_On(28) if $p->{value}->bit_test(31);
      $reg->[14]->Bit_On(29) if $ovfl;
      # m
      # i
   }

   return($mem_changed);
}


sub assemble {
   my($lines, $prg, $err) = @_;

   croak('usage: assemble(<code_aref>, <bin_aref>, <err_aref>)')
      unless
         ref($lines) eq 'ARRAY' and
         ref($prg)   eq 'ARRAY' and
         ref($err)   eq 'ARRAY';

   # first step: parse
   my $stmt = _parse_asm($lines, $err);

   # stop on errors
   foreach my $e ( @$err ) {
      return if $e->{level} eq 'error';
   }

   my(%bin, %labels, %substitute);
   my $addr = 0;
   my $lc   = 1;
   foreach my $s ( @$stmt ) {

      # skip empty lines
      next unless keys(%$s);

      # label addresses as hex strings
      $labels{$s->{label}} = sprintf("%08x", $addr) if $s->{label};

      if ( $_ = $s->{directive} ) {

         /\.ORG/ && do {
            $addr = $s->{operands}[0] =~ /^0/ ?
                    oct($s->{operands}[0])    :
                    $s->{operands}[0];
            next;
         };

         /\.DAT/ && do {
            my $value = $s->{operands}[0] =~ /^0/ ?
                        oct($s->{operands}[0])    :
                        $s->{operands}[0];
            $bin{sprintf("%08x", $addr++)} = sprintf("%08x", $value);
            next;
         };

         /\.BLK/ && do {
            my $len = $s->{operands}[0] =~ /^0/ ?
                      oct($s->{operands}[0])    :
                      $s->{operands}[0];
            for ( 1..$len ) {
               $bin{sprintf("%08x", $addr++)} = '0' x 8;
            }
            next;
         };

         /\.ASCII_B/ && do {
            my $fill = length($s->{operands}[0]) % 4;
            $s->{operands}[0] .= "\0" x (4 - $fill) if $fill;
            $bin{sprintf("%08x", $addr++)} = $_
               for reverse(
                  unpack("H*", reverse($s->{operands}[0])) =~ /.{8}/g
               );
            next;
         };

         /\.ASCII_L/ && do {
            $s->{operands}[0] .= "\0";
            $bin{sprintf("%08x", $addr++)} = sprintf("%08x", ord($_))
               for split(//, $s->{operands}[0]);
            next;
         }
      }

      if ( $s->{mnemonic} ) {

         my $itype = $Instr_Type{$s->{mnemonic}};

         my $instr = $Instructions{$itype}{INSTR}{$s->{mnemonic}};

         my $opc = @{$s->{operands}};

         unless ( $opc == $instr->{OPCOUNT} + 1 ) {
            push @$err, {
               line  => $lc,
               level => 'error',
               text  => "wrong number of operands for $s->{mnemonic} got $opc expected " . ($instr->{OPCOUNT} + 1),
            };
            next;
         }

         my @const;
         my $bin = 0;

         for my $i ( 0..2 ) {

            $s->{operands}[$i] ||= 'R0';
            my $parsed = {};

            unless ( _parse_op($s->{operands}[$i], $parsed) ) {
               push @$err, {
                  line  => $lc,
                  level => 'error',
                  text  => "syntax error in operand $i >>$s->{operands}[$i]<<",
               } ;
               next;
            }

            if ( $parsed->{const} ) {

               unless (
                  $parsed->{const} =~ /^0b[01]{1,32}$/i    or
                  $parsed->{const} =~ /^0x[0-9a-f]{1,8}$/i or
                  $parsed->{const} =~ /^\d{1,10}$/i
               ) {

                  # perhaps a label --> keep for third pass
                  $substitute{sprintf("%08x", $addr + @const + 1)} = {
                     line  => $lc,
                     label => $parsed->{const},
                  };

                  # preliminary code with 0
                  $parsed->{const} = '0x0';
               }

               push(@const, $parsed->{const});
            }

            my $op = $parsed->{const} ?
                     $parsed->{addreg} :
                     $parsed->{reg};
            $op  ||= 0;

            $op += 16 if $parsed->{id1} eq '--' or $parsed->{id2} eq '--';
            $op += 32 if $parsed->{id2} eq '++';
            $op += 48 if $parsed->{const};
            $op += 64 if $parsed->{indirect};

            $bin += $op << (7 * (2 - $i));
         }

         $bin += $instr->{OPCODE} << 21;

         if ( $itype eq 'HALT' ) {
            $bin += 6 << 25;
         } else {
            $bin += 4 << 25 if $itype eq 'SHIFT';
            if ( $s->{modifier} ) {
               $bin += 1 << 25 if $s->{modifier} =~ /M/i;
               $bin += 2 << 25 if $s->{modifier} =~ /C/i and $itype eq 'ALU';
            }
         }

         if ( $s->{cond_bit} ) {
            $bin += $SB_Nr{$s->{cond_bit}} << 28;
            $bin += 1 << 31 if $s->{cond_neg};
         }

         $bin{sprintf("%08x", $addr++)} = sprintf("%08x", $bin);

         $bin{sprintf("%08x", $addr++)} =
            sprintf("%08x", /^0/ ? oct($_) : $_) for @const;

      }
   } continue {
      $lc++;
   }

   for my $a ( keys(%substitute) ) {
      unless ( exists($labels{$substitute{$a}{label}}) ) {
         push @$err, {
            line  => $substitute{$a}{line},
            level => 'error',
            text  => "unknown operand $substitute{$a}{label}",
         };
         next;
      }
      $bin{$a} = $labels{$substitute{$a}{label}};
   }

   foreach my $e ( @$err ) {
      return if $e->{level} eq 'error';
   }

   @$prg = map { "$_ $bin{$_}" } sort keys(%bin);

   return(1);
}


sub _parse_asm {
   my($lines, $err) = @_;

   my (@statements, %vars, %labels, $stmt);

   my $lc = 1;
   for my $line ( @$lines ) {

      chomp($line);

      # waste removal
      $line =~ s/^\s+//;      # heading whitespace
      $line =~ s/[;*].*$//;   # comments

      $stmt = {};

      next unless $line =~ /\S/;  # skip empty lines

      $stmt->{operands} = [];

      # check for strings enclosed in '' as last parameter
      # if found replace whitespace with dummy string
      my ($head, $str) = $line =~ /^(.*?)'(.*)'\s*$/;
      if ( $str ) {
         $str =~ s/\s/$ws/g;
         $line = $head . $str;
      }

      # split into parts
      my @parts = split(/[,\s]\s*/, $line);

      # process the splitted parts backwards
      # first get operands until mnemonic or directive found
      my($opname, $mod) = $parts[-1] =~ /(\S+)\[(.*?)\]$/;
      $parts[-1] = uc($opname)
         if $opname and exists($Valid_Inst{uc($opname)});
      while (
         !exists($Valid_Inst{uc($parts[-1])}) and
         !exists($Valid_Dir{uc($parts[-1])})
      ) {
         unshift(@{$stmt->{operands}}, pop(@parts));
         last unless @parts;
         ($opname, $mod) = $parts[-1] =~ /(\S+)\[(.*?)\]$/;
         $parts[-1] = uc($opname)
            if $opname and exists($Valid_Inst{uc($opname)});
      }

      unless ( @parts ) {
         push @$err, {
            line  => $lc,
            level => 'error',
            text  => "no instruction or directive found",
         };
         next;
      }

      if ( $Valid_Dir{uc($parts[-1])} ) {

         # process directive statement
         # check basics
         unless ( (my $count = @{$stmt->{operands}}) == 1 ) {
            push @$err, {
               line  => $lc,
               level => 'error',
               text  => "wrong number of args got $count expected 1",
            };
            next;
         }

         $stmt->{directive} = uc(pop(@parts));

         if ( $mod ) {
            push @$err, {
               line  => $lc,
               level => 'error',
              text  => 'unexpected modifier after directive',
            };
            next;
         }

      } else {

         # process operation
         # basic checks
         $stmt->{mnemonic}  = uc(pop(@parts));

         if ( defined($mod) ) {
            if ( $mod !~ /^(?:C|M|CM|MC)$/i ) {
               push @$err, {
                  line  => $lc,
                  level => 'error',
                  text  => "unknown modifier $mod",
               };
               next;
            }

            $stmt->{modifier} = $mod;
         }

      }

      next unless @parts;

      # check for conditional execution
      my $sbits = join('', @Status_Bits);
      my($neg, $bit) = $parts[-1] =~ /^\?(!)?([$sbits])$/i;
      if ( $bit ) {
         $stmt->{cond_bit} = $bit;
         $stmt->{cond_neg} = $neg;
         pop(@parts);
      }

      # .EQU has to have a variable name
      unless ( @parts ) {
         if ( $stmt->{directive} and $stmt->{directive} eq '.EQU' ) {
            push @$err, {
               line  => $lc,
               level => 'error',
               text  => '.EQU directive without variable name',
            };
         }
         next;
      }

      # ... and it has to be unique
      if ( $stmt->{directive} and $stmt->{directive} eq '.EQU' ) {
         my $var = pop(@parts);
         if ( exists($vars{$var}) ) {
            push @$err, {
               line  => $lc,
               level => 'error',
               text  => "duplicate variable $var",
            };
            next;
         }

         $vars{$var} = $stmt->{operands}[0];

      } else {

         # only lables left
         my $label = pop(@parts);

         # already there?
         if ( exists($labels{$label}) ) {
            push @$err, {
               line  => $lc,
               level => 'error',
               text  => "duplicate label $label",
            };
            next;
         }

         $labels{$label} = 1;
         $stmt->{label}  = $label;

      }

      # nothing legal left
      if ( @parts ) {
         push @$err, {
            line  =>$lc,
            level => 'error',
            text  => "unexpected element(s) at begin of line",
         };
         next;
      }

   } continue {

      if ( $stmt->{operands} ) {
         OPER: for my $o ( @{$stmt->{operands}} ) {
            # substitute dummy with whitespace in operand strings
            $o =~ s/$ws/ /g;
            # substitute variables with values
            for my $v ( keys(%vars) )  {
               next OPER if $o =~ s/$v/$vars{$v}/;
            }
         }
      }

      # save parsed statement in array
      push(@statements, $stmt);
      ++$lc;
   }

   return(\@statements);
}


sub _parse_op {
   my($str, $parsed) = @_;

   @$parsed{qw/indirect id1 reg id2 const addreg/} =
      $str =~ /^
         (\@)?                                       # @
         (?:
            (?:
               (\-\-)?                               # --
               R([01]?(?(?<=1)[0-5]|[0-9]))          # R0-R15 or R00-R15
               (?(2)|(\+\+|\-\-)?)                   # ++|-- if not -- in \2
            )
            |
            (?:
               ((?:0[xb])?[0-9a-f]+|\S+?)            # dec, bin or hex number
               (?:\[R([01]?(?(?<=1)[0-5]|[0-9]))\])? # [R0-15]
            )
         )
      $/xi;

   for ( keys(%$parsed) ) {
      $parsed->{$_} = '' unless defined($parsed->{$_});
   }

   for ( values(%$parsed) ) {
      return(1) if defined($_);
   }

   return;
}


sub init_registers {
   my @regs;

   push(@regs, Bit::Vector->new_Hex(INSTR_SIZE, '0'))
      for 0..REGISTERS-1;

   $regs[0]  &= "0";
   $regs[14] |= "01000000";

   return(\@regs);
}


sub init_memory {
   my($size) = @_;
   croak('usage: init_memory(<size>)') unless $size;

   my @mem;
   push(@mem, Bit::Vector->new_Hex(INSTR_SIZE, '0'))
      for 0..$size-1;

   return(\@mem);
}


sub init_code {
   my($mem_ref) = @_;

   croak('usage: init_memory(<mem_aref>)')
      unless ref($mem_ref) eq 'ARRAY';

   my @code = map { '' } 0..$#{$mem_ref};

   return(\@code);
}


sub loadbin {
   my($mem_ref, $bin_ref) = @_;

   croak('usage: init_memory(<mem_aref>, <bin_aref>)')
      unless
         ref($mem_ref) eq 'ARRAY' and
         ref($bin_ref) eq 'ARRAY';

   chomp(@$bin_ref);

   for my $bin ( @$bin_ref ) {
      my($addr, $data) = split(/\s+/, $bin);
      $mem_ref->[hex($addr)]->from_Hex($data);
   }
}


sub _parse_instr {
   my($instr) = @_;

   my %pi;

   # parse instruction
   $pi{negate}                = $instr->bit_test(31);
   $pi{cond}                  = $instr->Chunk_Read(3, 28);
   @pi{qw/mod use_carry alu/} = map { $instr->bit_test($_) } (25..27);
   $pi{opcode}                = $instr->Chunk_Read(4, 21);
   $pi{params}                = [
      map { {
         addr => $instr->Chunk_Read(3, 18 - $_ * 7),
         reg  => $instr->Chunk_Read(4, 14 - $_ * 7),
      } } (0..2)
   ];

   # determine instruction type and operator count
   $pi{instr_type} = $pi{alu} ?
                     (
                        $pi{use_carry} ?
                        $Instructions{HALT}{TYPE} :
                        $Instructions{SHIFT}{TYPE}
                     ) :
                     $Instructions{ALU}{TYPE};

   $pi{type_name}  = $Instr_TypeNr{$pi{instr_type}};
   $pi{op_name}    = $pi{type_name} eq 'HALT' ?
                     'HALT'                   :
                     $Opcode_Names{$pi{type_name}}{$pi{opcode}};

   $pi{opcount}    = $Instructions{$pi{type_name}}{INSTR}{$pi{op_name}}{OPCOUNT};

   return(\%pi);
}

1;

__END__

=head1 NAME

NICE - a module to emulate, assemble and disassemble code for the
NICE processor (NICE Is Charmingly Elegant)

=head1 VERSION

This documentation describes the first fully functional version 0.4.

=head1 SYNOPSIS

  # load module, import all functions
  use NICE qw/:all/;

  # some code as line array
  my @code = split /\n/, qq{
    MOVE R1 R0
    MOVE R1 0x125690CD
    HALT
  };

  # arrays for results, passed by reference
  my @bin = ();
  my @errors = ();

  # assemble
  if ( assemble(\@code, \@bin, \@errors) ) {
     print "$_\n" for @bin;
  } else {
     print "$_->{level}: $_->{text} line $_->{line}\n"
        for @errors;
  }

  # init memory and registers
  my $mem     = init_memory(0x800);
  my $reg     = init_registers();
  my $changed = {};

  # load assembled code into memory
  loadbin($mem, \@bin);

  # run program, dump data
  do {
     printf "%08X %08X\n", $_, $reg->[$_] for @$reg;
     print "\n";
     printf "%08X %08X\n", $_, $changed->{$_}
        for sort keys(%$changed);
  } until ( ! step($mem, $reg, $changed) );

  # disassemble from memory
  my @dcode = init_code($mem);
  disassemble($mem, \@dcode);

  printf "%08X %08X\n", $_, $dcode->[$_]
     for 0..$#{$mem};

=head1 DESCRIPTION

The NICE module provides functions to assemble, disassemble and run programs
written for the NICE processor, an experimental processor designed by Bernd
Ulmann <http://www.vaxman.de>.

An overview of the properties can be found at:

L<http://www.vaxman.de/projects/nice/nice.html>

=head1 FUNCTIONS

=head2 C<< init_registers() >>

Initializes an array with the 16 32-bit registers of the NICE processor
(implemented as Bit::Vector objects). You will need them for executing
instructions with the step function.

The return value is a reference to the registers array.

=head2 C<< init_memory(<size>) >>

Initializes a memory array (an array of Bit::Vector objects) for subsequent
use with the step or disassembler function. Size is the number of 32bit
words to inlitialize.

The return value is a reference to the memory array.

=head2 C<< init_code(<mem_aref>) >>

Initializes an array for storing disassembled code in. By passing it the
array reference pointing at the memory, you will get an array with the same
number of elements.

The return value is a reference to the code array.

=head2 C<< loadbin(<mem_aref>, <bin_aref>) >>

Loads the binary data from the array referenced by bin_aref into the memory
array referenced by mem_aref.

=head2 C<< assemble(<coderef>, <binref>, <errorref>) >>

The assemble function assembles (Surprise, Surprise!) the NICE assembler code
given in an array of lines. The result will be put in two corresponding arrays
for the resulting binary (if the code could be assembled correctly) and the
errors encountered.

The array elements of the binary array will contain one instruction or data word
in NICE binary format. This is a simple string with a memory address and the
instruction (8 hex nibbles each) separated by a space.

The array elements of the error array will be hashrefs referencing a hash with
the structure:

   {
      level => <the errorlevel: 'error', 'warning' or 'info'>,
      text  => <the error description>,
      line  => <the line number, where the error occured>,
   }

In case of failure (one ore more errors with level 'error') the function will
return undef and the code array will not be filled. Otherwise it will return a
true value.

=head2 C<< disassemble(<mem_aref>, <code_aref>, [<changed_mem_href>]) >>

Disassembles code located in a memory array to a corresponding code array.
You can disassemble instructions from specific memory locations by
suppliying an additional hash with the memory addresses as keys.

This is escpecially useful in combination with the step function, that
fills a reference to a hash with the memory addresses as keys and their
instructions as values containig only those addresses changed by the last
executing cycle.

=head2 C<< step(<mem_aref>, <reg_aref>, <changed_mem_href>) >>

The step function executes (makes the appropriate changes to registers and
memory for) the one instruction pointed to by the contents of the program
counter in register 15.

If memory data is changed, the addresses and their values are put into the
hash referenced by the third parameter. This is useful for displying data
without the need for unnessesary refreshing (see disassemble).

It will return a false value if the instruction was a HALT instruction.
Otherwise it will return a true value.

=head1 TODO

=over

=item - disassemble from file, filehandle or scalar

=item - assemble from file, filehandle or scalar

=item - object interface?

=back

There is no schedule for implementation

=head1 BUGS

Error checking in case of buggy or malformed data could be better.

=head1 AUTHOR

Bernd Ulmann <ulmann@vaxman.de>, Thomas Kratz <tkr@gnlpf.de>

=head1 COPYRIGHT

Copyright (c) 2004 Bernd Ulmann & Thomas Kratz. All rights reserved.
This program is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=cut
