use strict;
use warnings;

use FindBin qw/$Bin/;
use lib $Bin;

use Tk;

use Time::HiRes qw/time/;

use File::Basename;
use File::Spec::Functions;

use NICE qw/:all/;

=for TODO

   * options dialog <-> cfg file

  Code Editor:
   * overstrike mode with box cursor ?

=cut

## INIT
$| = 1;

# script scoped lexicals

my $title = 'NICE_IDE';

my $menufont;
my %W;
my %HexEntry = (
   Value   => undef,
   Display => '',
   ToWdg   => undef,
   ToAddr  => undef,
);
my $Interrupt = 0;
my $Memory;
my $Code;
my $Registers;
my %Watches;
my $RunBtTxt = 'Run [F5]';
my $PC = 0;
my @CodeCache;
my @MemCache;
my @RegCache;
my $last_run = 0;
my $PopWidget;
my $PopCell;
my $EditOpenCmd;
my $LastBinFile = '';

my $default_font = $^O eq 'MSWin32' ? 'Fixedsys' : 'Courier 10';

# config values
my %cfg = (
   ButtonWidth  => 8,
   RunSpeed     => 1,
   AppSize      => 0.8,
   MemSize      => 0x1000,
   OutputWin    => 0x400,
   OutputX      => 80,
   OutputY      => 25,
   IgnoreOutput => 1,
   InstrSteps   => {map { $_ => 1 } qw/6 13 20 24 27 31/},
   WatchSize    => 30,
   MemoryFont   => $default_font,
   RegisterFont => $default_font,
   WatchFont    => $default_font,
   HexEntryFont => $default_font,
   CodeLineFont => 'Courier 9 bold',
   EditorFont   => $default_font,
   OutputFont   => $default_font,
   MemHeader    => [qw/ADDR VALUE CODE/],
   RegHeader    => [qw/REG VALUE/],
   WatchHeader  => [qw/WATCH VALUE/],
   PcColor      => '#FF8080',
);
$cfg{InstrVar}  = [map { 0  } (0..NICE::INSTR_SIZE-1) ];
$cfg{InstrCode} = [map { '' } keys(%{$cfg{InstrSteps}})];
$cfg{OutputEnd} = $cfg{OutputWin} + $cfg{OutputX} * $cfg{OutputY} - 1;

# editor key bindings
my %ed_bindings = qw(
   File/Open         <Control-o>
   File/Save         <Control-s>-
   Edit/Copy         <Control-Insert>
   Edit/Cut          <Control-Delete>
   Search/Find       <Control-f>
   Search/Find_Next  <F3>
   Search/Replace    <Control-r>-
   View/Goto_Line... <Control-g>-
   &DeleteLine       <Control-y>
);

my %ed_functions = (
   DeleteLine => sub {
      my($ed) = @_;
      $ed->delete('insert linestart', 'insert linestart + 1 lines');
   },
);

# create main window
$W{top} = MainWindow->new( -title => $title );
$W{top}->iconify;
my($tx, $ty) = WinResize($W{top}, $cfg{AppSize});

# Window Icon
my $icondata;
{
   local $/;
   $icondata = <DATA>;
}
my $icon = $W{top}->Pixmap(-data => $icondata);
$W{top}->idletasks;
$W{top}->iconimage($icon);

$W{top}->after(200, \&Startup);

MainLoop();

## SUBS

sub Startup {

   require Tk::SplitFrame;
   require Tk::LabFrame;
   require Tk::Scale;
   require Tk::ROText;
   require Tk::TableMatrix;
   require Tk::ItemStyle;
   require Tk::NoteBook;
   require Tk::TextUndo;
   require Tk::LineNumberText;
   require Tk::Font;

   CreateMenus();
   $menufont = $W{popup}->cget('-font');

   # Notebooks
   $W{nb} = $W{top}->NoteBook(
      -font        => $menufont,
      -borderwidth => 1,
   )->form(qw/-t %0 -b %100 -l %0 -r %100/);
   $W{nb}->add(
      $_,
      -label    => $_,
      -raisecmd => [\&NbRaise, $_],
   ) for (qw/Debugger Editor/);

   # Editor Page
   $W{sp_ed} = $W{nb}->page_widget('Editor')->SplitFrame(qw/
      -orientation    horizontal
      -trimcolor      grey
      -background     white
      -sliderwidth    2
      -relief         sunken
      -height         100
      -width          100
      -padbefore      0
      -padafter       0
   /);
   $W{fr_ed} = $W{sp_ed}->Frame(
      -relief      => 'sunken',
      -borderwidth => 0,
   );
   $W{ro_status} = $W{sp_ed}->Scrolled(
      'ROText',
      -scrollbars  => 'osoe',
      -font        => $cfg{EditorFont},
      -height      => 5,
      -bg          => 'white',
      -borderwidth => 0,
   );
   $W{ro_status}->menu->configure(-bg => $W{popup}->cget('-bg'));

   $W{sp_ed}->form(qw/-l %0 -t %0 -r %100 -b %100/);

   $W{fr_ec} = $W{fr_ed}->Frame(
      -relief      => 'sunken',
      -borderwidth => 0,
   )->form(qw/-t %0 -l %0 -r %100/);

   CreateEditControlPane($W{fr_ec});

   $W{ct_edit} = $W{fr_ed}->LineNumberText(
      -widget      => 'TextUndo',
      -wrap        => 'none',
      -scrollbars  => 'osoe',
      -background  => 'grey90',
      -font        => $cfg{EditorFont},
      -borderwidth => 1,
   )->form(-t => [$W{fr_ec}], qw/-b %100 -l %0 -r %100/);

   # editor menu as main menu
   $W{top}->configure(-menu => $W{ct_edit}->menu());
   my $ecb = $W{ct_edit}->menu->entrycget(
      'File', '-menu',
   )->entrycget('Open', -command);
   $EditOpenCmd = sub {
      $W{top}->configure(
         -title => $title. ' - ' . ($W{ct_edit}->FileName() || ''),
      );
      print "No widget!\n" unless $ecb;
      $ecb->Call();
   };

   ModEditorBindings($W{ct_edit}->Subwidget('scrolled'), \%ed_bindings);

   # debugger page

   $W{top}->bind(
      '<Control-o>',
      sub {
         OpenFile() if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{top}->bind(
      '<Control-l>',
      sub {
         OpenFile($LastBinFile) if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{sp_lr} = $W{nb}->page_widget('Debugger')->SplitFrame(qw/
      -orientation    vertical
      -trimcolor      grey
      -background     white
      -sliderwidth    2
      -relief         sunken
      -height         100
      -width          100
      -padbefore      0
      -padafter       0
   /);

   $W{fr_cm} = $W{sp_lr}->Frame(
      -relief => 'sunken',
   );
   $W{sp_rw} = $W{sp_lr}->SplitFrame(qw/
      -orientation    horizontal
      -trimcolor      grey
      -background     white
      -sliderwidth    2
      -relief         sunken
      -height         100
      -width          100
      -padbefore      0
      -padafter       0
   /);

   my $but = CreateControlPane(
      $W{fr_cm},
      -l => '%0',    -t => '%0',
      -r => '%100',
   );

   CreateMemoryPane(
      $W{fr_cm}, -t => [$but],
      qw/-l %0 -r %100 -b %100/
   );

   $W{sp_lr}->form(qw/-l %0 -t %0 -r %100 -b %100/);

   my $reg = CreateRegisterPane(
      $W{sp_rw},
      -l => '%0',   -t => '%0',
      -r => '%100',
      qw/-padx 0 -pady 0/
   );

   my $wat = CreateWatchPane(
      $W{sp_rw},
      -l => '%0',   -t => [$reg],
      -r => '%100', -b => '%100',
      qw/-padx 0 -pady 0/
   );

   $W{sp_rw}->form(qw/-l %0 -t %0 -r %100 -b %100/);

   RegistersHeaders();
   MemoryHeaders();

   RegistersInit();
   MemoryInit();
   CodeInit();

   MemoryDisplay();
   RegistersDisplay();

   $W{top}->update;

   my $topw = $W{top}->width;
   my $toph = $W{top}->height;
   my $regw = $W{tmreg}->Subwidget('scrolled')->reqwidth  + 28;
   my $regh = $W{tmreg}->Subwidget('scrolled')->reqheight + 4;

   $W{sp_lr}->configure(-sliderposition => $topw - $regw);
   $W{sp_rw}->configure(-sliderposition => $regh);
   $W{sp_ed}->configure(-sliderposition => $topw * 0.6);

   $W{top}->deiconify;
   WinResize($W{top}, $cfg{AppSize} + 0.01);
   $W{top}->update;

}


sub CreateMenus {

   # Popup for Button-3
   $W{popup} = $W{top}->Menu(-tearoff => 0);
   $W{popup}->command(
      -label   => 'Add ~Watch',
      -command => [\&ModWatch, 'add'],
   );
   $W{popup}->command(
      -label   => 'Delete ~Watch',
      -command => [\&ModWatch, 'delete'],
   );
   $W{popup}->separator();
   $W{popup}->command(
      -label   => '~Clear',
      -command => [\&ClearPane],
   );
   $W{popup}->separator();
   $W{popup}->command(
      -label   => 'Clear ~All',
      -command => [\&ClearPane, 'all'],
   );

}


sub CreateMemoryPane {
   my $parent = shift;

   $W{tmmem} = $parent->Scrolled(
      'TableMatrix',
      -font           => $cfg{MemoryFont},
      -cache          => 1,
      -cols           => scalar(@{$cfg{MemHeader}}),
      -colwidth       => NICE::INSTR_SIZE / 4,
      -colstretchmode => 'last',
      -rows           => $cfg{MemSize} + 1,
      -flashmode      => 1,
      -flashtime      => 2,
      -selectmode     => 'single',
      -titlerows      => 1,
      -titlecols      => 1,
      -anchor         => 'w',
      -scrollbars     => 'osoe',
      -padx           => 5,
   )->form(@_);

   $W{tmmem}->bind(
      '<Double-Button-1>',
      [\&LClick, 'double', Ev('x'), Ev('y')],
   );
   $W{tmmem}->bind(
      '<Button-1>',
      [\&LClick, 'single', Ev('x'), Ev('y')],
   );
   $W{tmmem}->bind(
      '<Button-3>',
      [\&RClick, Ev('X'), Ev('Y'), Ev('x'), Ev('y')],
   );

   $W{tmmem}->tagConfigure(
      'flash',
      -fg    => 'red',
      -bg    => 'grey90',
      -state => 'disabled',
   );
   $W{tmmem}->tagConfigure(
      'normal',
      -bg    => 'grey90',
      -state => 'disabled',
   );
   $W{tmmem}->tagConfigure(
      'title',
      -bg     => 'grey60',
      -state  => 'disabled',
      -relief => 'sunken',
      -font   => $cfg{MemoryFont},
   );
   $W{tmmem}->tagConfigure(
      'pc',
      -bg    => $cfg{PcColor},
      -state => 'disabled',
   );

   $W{tmmem}->tagRaise('pc', 'normal');
   $W{tmmem}->tagCol('normal', 1, 2);
   $W{tmmem}->activate('1,1');

   return($W{tmmem});
}


sub CreateRegisterPane {
   my $parent = shift;

   $W{tmreg} = $parent->Scrolled(
      'TableMatrix',
      -font           => $cfg{RegisterFont},
      -cache          => 1,
      -cols           => scalar(@{$cfg{RegHeader}}),
      -colwidth       => NICE::INSTR_SIZE / 4,
      -rows           => NICE::REGISTERS + 2,
      -colstretchmode => 'last',
      -flashmode      => 1,
      -flashtime      => 1,
      -selectmode     => 'single',
      -titlerows      => 1,
      -titlecols      => 1,
      -anchor         => 'w',
      -scrollbars     => 'osoe',
      -padx           => 5,
   )->form(@_);

   $W{tmreg}->tagConfigure(
      'flash',
      -fg => 'red',
      -bg => 'grey90',
      -state => 'disabled',
   );
   $W{tmreg}->tagConfigure(
      'normal',
      -bg => 'grey90',
      -state => 'disabled',
   );
   $W{tmreg}->tagConfigure(
      'title',
      -bg     => 'grey60',
      -state  => 'disabled',
      -relief => 'sunken',
      -font   => $cfg{RegisterFont},
   );

   $W{tmreg}->bind(
      '<Double-Button-1>',
      [\&LClick, 'double', Ev('x'), Ev('y')],
   );
   $W{tmreg}->bind(
      '<Button-1>',
      [\&LClick, 'single', Ev('x'), Ev('y')],
   );
   $W{tmreg}->bind(
      '<Button-3>',
      [\&RClick, Ev('X'), Ev('Y'), Ev('x'), Ev('y')],
   );

   $W{tmreg}->tagCol('normal', 1);
   $W{tmreg}->activate('1,1');

   return($W{tmreg});
}


sub CreateWatchPane {
   my $parent = shift;

   $W{tmwatch} = $parent->Scrolled(
      'TableMatrix',
      -font           => $cfg{WatchFont},
      -cache          => 1,
      -cols           => scalar(@{$cfg{WatchHeader}}),
      -colwidth       => NICE::INSTR_SIZE / 4,
      -colstretchmode => 'last',
      -rows           => 1,
      -flashmode      => 1,
      -flashtime      => 1,
      -selectmode     => 'single',
      -titlerows      => 1,
      -titlecols      => 1,
      -anchor         => 'w',
      -scrollbars     => 'osoe',
      -padx           => 5,
   )->form(@_);

   $W{tmwatch}->tagConfigure(
      'flash',
      -fg => 'red',
      -bg => 'grey90',
      -state => 'disabled',
   );
   $W{tmwatch}->tagConfigure(
      'normal',
      -bg => 'grey90',
      -state => 'disabled',
   );
   $W{tmwatch}->tagConfigure(
      'title',
      -bg     => 'grey60',
      -state  => 'disabled',
      -relief => 'sunken',
      -font   => $cfg{WatchFont},
   );

   $W{tmwatch}->set("0,$_", $cfg{WatchHeader}[$_]) for (0..$#{$cfg{WatchHeader}});

   $W{tmwatch}->tagCol('normal', 1);

   $W{tmwatch}->bind(
      '<Button-3>',
      [\&RClick, Ev('X'), Ev('Y'), Ev('x'), Ev('y')],
   );

   return($W{tmwatch});
}


sub CreateControlPane {
   my $parent = shift;

   $W{frctrl} = $parent->Frame(
      -relief => 'sunken',
   )->form(@_);

   $W{frBut} = PLabFrame($W{frctrl}, 'Action');
   $W{frBut}->form(qw/-l %0 -t %0 -pady 0/);

   $W{btRun} = $W{frBut}->Button(
      -width        => $cfg{ButtonWidth},
      -textvariable => \$RunBtTxt,
      -command      => sub {
         if ( $RunBtTxt eq 'Run [F5]' ) {
            $RunBtTxt = 'Stop [F5]';
            RunProgram('continuous');
         } else {
            $RunBtTxt = 'Run [F5]';
         }
      },
   )->form(qw/-t %0 -l %0 -tp 2 -lp 2/);
   $W{top}->bind(
      '<F5>',
      sub {
         $W{btRun}->invoke() if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{btStep} = $W{frBut}->Button(
      -width   => $cfg{ButtonWidth},
      -text    => 'Step [F6]',
      -command => [\&RunProgram, 'single'],
   )->form(
      -l => [$W{btRun}], -t => ['&', $W{btRun}],
      -tp => 2, -lp => 2,
   );
   $W{top}->bind(
      '<F6>',
      sub {
         $W{btStep}->invoke() if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{btIntr} = $W{frBut}->Button(
      -text    => 'Intr [F7]',
      -command => sub { $Interrupt = 1; },
      -width   => $cfg{ButtonWidth},
   )->form(
      -l => [$W{btStep}], -t => ['&', $W{btStep}],
      -tp => 2, -lp => 2,
   );
   $W{top}->bind(
      '<F7>',
      sub {
         $W{btIntr}->invoke() if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{frEdit} = PLabFrame($W{frctrl}, 'Edit');
   $W{frEdit}->form(
      -l => [$W{frBut}],      -t    => ['&', $W{frBut}],
      -b => ['&', $W{frBut}], -pady => 0,
   );

   $W{enhex} = $W{frEdit}->Entry(
      -textvariable    => \$HexEntry{Value},
      -font            => $cfg{HexEntryFont},
      -width           => 10,
      -validate        => 'key',
      -validatecommand => sub {
         return unless $_[0] =~ /^[0-9a-f]{0,8}$/i;
         $HexEntry{value} = hex($_[0]);
         for my $i ( 0..NICE::INSTR_SIZE-1 ) {
            $cfg{InstrVar}[$i] = 0 + (($HexEntry{value} & 2**$i) != 0);
         }
         ShowInstructionCode($HexEntry{value});
         return(1);
      },
   )->form(qw/-t %0 -l %0 -tp 2 -lp 2/);

   $W{btSet} = $W{frEdit}->Button(
      -text    => '-->',
      -font    => $cfg{HexEntryFont},
      -command => [\&SetHexValue],
      -width   => 4,
      -padx    => 0,
      -pady    => 0,
      -borderwidth => 1,
      -highlightthickness => 0,
   )->form(
      -l => [$W{enhex}], -t => ['&', $W{enhex}],
      qw/-tp 2/,
   );
   $W{top}->bind(
      '<Return>',
      sub {
         $W{btSet}->invoke() if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{lbhex} = $W{frEdit}->Label(
      -text => sprintf("%3s[%8s]", '', ''),
      -font => $cfg{HexEntryFont},
   )->form(
      -l => [$W{btSet}], -t => ['&', $W{btSet}]
   );

   $W{frSpeed} = PLabFrame($W{frctrl}, 'Speed');
   $W{frSpeed}->form(
      -l => [$W{frEdit}],      -t    => ['&', $W{frEdit}],
      -b => ['&', $W{frEdit}], -pady => 0,
   );

   $W{lbspeed} = $W{frSpeed}->Label(
      -text => 'Speed [Hz]:',
   )->form(qw/-t %0 -l %0 -tp 2 -lp 2/);

   $W{spspeed} = $W{frSpeed}->Spinbox(
      -textvariable => \$cfg{RunSpeed},
      -from         => 1,
      -to           => 500,
      -width        => 4,
   )->form(
      -l => [$W{lbspeed}], -t => ['&', $W{lbspeed}],
      qw/-lp 2 -tp 2/,
   );
   $W{top}->bind(
      '<Up>',
      sub {
         $W{spspeed}->invoke('buttonup') if $W{nb}->raised() eq 'Debugger';
      }
   );
   $W{top}->bind(
      '<Down>',
      sub {
         $W{spspeed}->invoke('buttondown') if $W{nb}->raised() eq 'Debugger';
      }
   );

   $W{frInstr} = PLabFrame($W{frctrl}, 'Instruction');
   $W{frInstr}->form(
      -l => '%0', -t => [$W{frBut}],
      -pady => 0,
   );


   my $yspace = 29;
   for my $i ( 0..NICE::INSTR_SIZE-1 ) {
      my $bit = NICE::INSTR_SIZE - $i - 1;
      $W{btInstr}[$bit] = $W{frInstr}->Button(
         -textvariable       => \$cfg{InstrVar}[$bit],
         -font               => $cfg{HexEntryFont},
         -command            => [\&ToggleInstrBit, $bit],
         -width              => 1,
         -pady               => 0,
         -padx               => 0,
         -bg                 => 'grey95',
         -highlightthickness => 0,
      )->form(
         -l  => $i == 0 ? '%0' : [$W{btInstr}[$bit+1]],
         -t  => '%0',
         -lp => $cfg{InstrSteps}{$bit} ? $yspace : 0,
      );
   }

   my $i = 0;
   for my $bit ( sort { $b <=> $a } keys(%{$cfg{InstrSteps}}) ) {
      $W{lbInstrCode}[$i] = $W{frInstr}->Label(
         -textvariable => \$cfg{InstrCode}[$i++],
         -font         => $cfg{CodeLineFont},
      )->form(
         -l  => ['&', $W{btInstr}[$bit]],
         -t  => [$W{btInstr}[0]],
         -lp => $yspace,
      );
   }

   return($W{frctrl});
}


sub CreateEditControlPane {
   my($w) = @_;

   $W{frEdBut} = PLabFrame($w, 'Actions');
   $W{frEdBut}->form(
      qw/-l %0 -t %0 -r %100 -b %100/
   );

   $W{btAsm} = $W{frEdBut}->Button(
      -width   => $cfg{ButtonWidth},
      -text    => 'Asm [F2]',
      -command => [\&Assemble],
   )->form(qw/-l %0 -t %0 -lp 2/);
   $W{top}->bind(
      '<F2>',
      sub {
         $W{btAsm}->invoke() if $W{nb}->raised() eq 'Editor';
      }
   );

   $W{btDeb} = $W{frEdBut}->Button(
      -width   => $cfg{ButtonWidth},
      -text    => 'Debug [F4]',
      -command => sub {
         $W{ct_edit}->menu->entrycget(
            'File', '-menu',
         )->invoke('Save');
         my $binfn = Assemble();
         return unless $binfn;
         $W{nb}->raise('Debugger');
         OpenFile($binfn);
      },
   )->form(-l => [$W{btAsm}], qw/-t %0 -lp 2/);
   $W{top}->bind(
      '<F4>',
      sub {
         $W{btDeb}->invoke() if $W{nb}->raised() eq 'Editor';
      }
   );

}


sub RunProgram {
   my($mode) = @_;

   return if $mode eq 'continuous' and $RunBtTxt eq 'Run [F5]';

   my $changed = {};
   my $rc = step($Memory, $Registers, $changed);

   MemoryDisplay($changed);
   RegistersDisplay();

   return unless $mode eq 'continuous';

   unless ( $rc ) {
      $RunBtTxt = 'Run [F5]';
      return;
   }

   $cfg{RunSpeed} ||= 1;

   $W{top}->after(
      1000/$cfg{RunSpeed},
      [\&RunProgram, 'continuous']
   );
}


sub SetHexValue {

   return unless defined($HexEntry{Value}) and defined($HexEntry{ToWdg});

   if ( $HexEntry{ToWdg} eq $W{tmreg}->Subwidget('scrolled') ) {

      $Registers->[$HexEntry{ToAddr}]->from_Hex($HexEntry{Value});
      RegistersDisplay();

   } else {

      $Memory->[$HexEntry{ToAddr}]->from_Hex($HexEntry{Value});
      MemoryDisplay({
         $HexEntry{ToAddr} => $Memory->[$HexEntry{ToAddr}],
      });

   }
}


sub WinResize {
   my($w, $s) = @_;
   my($maxx, $maxy) = $w->maxsize();
   my $newx = int($maxx * $s);
   my $newy = int($maxy * $s);
   my $left = $maxx - $newx - 8;
   my $top  = $maxy - $newy - 48;
   $w->geometry("${newx}x$newy+$left+$top");
   return($newx, $newy);
}


sub MemoryInit {
   $Memory = init_memory($cfg{MemSize});
}

sub CodeInit {
   $Code   = init_code($Memory);
}

sub RegistersInit {
   $Registers = init_registers();
}


sub RegistersHeaders {

   $W{tmreg}->set("0,$_", $cfg{RegHeader}[$_]) for (0..$#{$cfg{RegHeader}});

   $W{tmreg}->set("$_,0", sprintf("R%02s", $_ - 1)) for (1..NICE::REGISTERS);

   $W{tmreg}->set(NICE::REGISTERS + 1 . ',0', 'SR');
}


sub MemoryHeaders {

   $W{tmmem}->set("0,$_", $cfg{MemHeader}[$_]) for (0..$#{$cfg{MemHeader}});

   $W{tmmem}->set("$_,0", sprintf("%08X", $_ - 1)) for (1..$cfg{MemSize});
}


sub MemoryDisplay {
   my($mem_changed) = @_;

   my @mem_update = ref($mem_changed) eq 'HASH' ?
                    keys(%$mem_changed)         :
                    0..$#{$Memory}              ;

   my $code_changed = disassemble($Memory, $Code, $mem_changed);

   for my $addr ( @mem_update ) {

      $W{tmmem}->set(($addr+1) . ',1', $Memory->[$addr]);

      $W{tmwatch}->set("$Watches{$addr},1", $Memory->[$addr])
         if exists($Watches{$addr});

      next unless ref($mem_changed) eq 'HASH';

      UpdateOutput($addr)
         if defined($cfg{OutputWin}) and
            $addr >= $cfg{OutputWin} and
            $addr <= $cfg{OutputEnd};
   }

   for my $addr ( keys(%$code_changed) ) {
      $W{tmmem}->set(($addr+1) . ',2', $Code->[$addr]);
   }

   # set PC
   my $idx = $Registers->[15]->to_Dec();
   $W{tmmem}->tagRow('normal', $PC+1);
   $W{tmmem}->tagRow('pc', $idx+1);
   $PC = $idx;

   $W{tmmem}->see("$idx,0");
}


sub RegistersDisplay {

   for my $i (0..NICE::REGISTERS-1) {

      next if defined($RegCache[$i]) and $RegCache[$i] == $Registers->[$i];

      $W{tmreg}->set(($i+1).',1', $Registers->[$i]);

      if ( $i == NICE::REGISTERS-2 ) {

         my $sbtxt;
         for my $j ( 0..$#Status_Bits ) {
            $sbtxt .= $Registers->[$i]->bit_test(NICE::INSTR_SIZE - 1 - $j) ?
                      $Status_Bits[7 - $j] :
                      '_';
         }

         $W{tmreg}->set((NICE::REGISTERS+2).',1', $sbtxt);
      }
   }

   @RegCache = map { $_->Clone() } @$Registers;
}


sub OpenFile {
   my($fn) = @_;

   $fn ||= $W{top}->getOpenFile(
      -filetypes  => [
         ['Binaries', '.bin'],
         $^O eq 'MSWin32' ? [] : (),
      ],
      -initialdir => $Bin,
      -title      => "Choose File",
   );

   return unless $fn;

   my $fh;
   unless ( open($fh, '<', $fn) ) {
      ErrMsg("error opening >>$fn<<, $!");
      return;
   }

   MemoryInit();

   my @bin = <$fh>;
   loadbin($Memory, \@bin);

   RegistersInit();
   RegistersDisplay();
   MemoryDisplay();

   $W{top}->configure(-title => $title. ' - ' . $fn);
   $LastBinFile = $fn;

   $W{Output}->DESTROY if exists $W{Output};
}


sub ToggleBreakPoint {

   my $addr = ($W{limem}->infoSelection())[0];
   return unless $addr;

   my $mem = $W{limem}->infoData($addr);

}


sub LClick {
   my($w, $mode, $x, $y) = @_;

   my($addr) = $w->index("\@$x,$y", 'row');
   --$addr;

   return if $addr < 0;

   my $value = $w->get('active');

   my $lab = $w eq $W{tmreg}->Subwidget('scrolled') ? 'REG' : 'MEM';

   $HexEntry{ToWdg}  = $w;
   $HexEntry{ToAddr} = $addr;

   $W{lbhex}->configure(
      -text => sprintf("%3s[%08X]", $lab, $addr),
   );

   return unless $value =~ /^[0-9A-F]{8}$/;

   if ( $mode eq 'double' ) {
      $HexEntry{Value} = $value;
      $W{enhex}->focus();
      $W{enhex}->icursor('end');
   }
}


sub RClick {
   my($w, $X, $Y, $x, $y) = @_;

   $W{popup}->entryconfigure(
      'Add Watch',
      -state => $w eq $W{tmmem}->Subwidget('scrolled') ?
                'normal' : 'disabled',
   );
   $W{popup}->entryconfigure(
      'Delete Watch',
      -state => $w eq $W{tmwatch}->Subwidget('scrolled') ?
                'normal' : 'disabled',
   );

   $PopWidget = $w;
   $PopCell   = "\@$x,$y";

   $W{popup}->post($X, $Y);
}


sub ClearPane {
   my($mode) = @_;

   $mode ||= '';
   foreach ( qw/mem watch reg/ ) {
      my $w = $W{'tm' . $_}->Subwidget('scrolled');
      if ( $mode eq 'all' or $w eq $PopWidget ) {
         /mem/ && do {
                     MemoryInit();
                     MemoryDisplay();
                     next;
                  };
         /watch/ && do {
                     %Watches = ();
                     my $last = $W{tmwatch}->index('end', 'row');
                     $W{tmwatch}->deleteRows(1, $last);
                     next;
                  };
         /reg/ && do {
                     RegistersInit();
                     RegistersDisplay();
                     next;
                  };
      }
   }
}


sub ModWatch {
   my($mode) = @_;

   my($addr) = $PopWidget->index($PopCell, 'row');

   return if $addr < 1;

   if ( $mode eq 'add' ) {

      return if exists($Watches{$addr-1});
      # TODO MsgBox 'already exists';

      $W{tmwatch}->insertRows('end', 1);

      my $watch_addr = $W{tmwatch}->index('end', 'row');
      $Watches{$addr-1} = $watch_addr;

      for (0..1) {
         my $v = $PopWidget->get("$addr,$_");
         $W{tmwatch}->set("$watch_addr,$_", $v);
      }

   } else {

      my $mem_addr = $W{tmwatch}->get("$addr,0") + 0;
      delete($Watches{$mem_addr});

      $W{tmwatch}->deleteRows($addr, 1);
   }
}


sub NbRaise {
   my($name) = @_;

   my $m = $W{ct_edit}->menu();

   # enable and disable appropriate menu entries
   my $state = $name eq 'Editor' ? 'normal' : 'disabled';

   foreach my $i ( 0..$m->index('end') ) {
      $m->entryconfigure($i, -state => $state)
         unless $m->entrycget($i, '-label') eq 'File';
   }

   # also for entries in 'File' menu
   $m = $m->entrycget('File', '-menu');

   foreach my $i ( 0..$m->index('end') ) {
      next if $m->type($i) eq 'separator';
      my $lb = $m->entrycget($i, '-label');
      $m->entryconfigure($i, -state => $state)
         unless $lb eq 'Open' or $lb eq 'Exit';
   }

   # redirecting Open command, setting focus and title
   if ( $name eq 'Editor' ) {
      $m->entryconfigure('Open', -command => $EditOpenCmd);
      $W{ct_edit}->focus;
      $W{top}->configure(
         -title => $title. ' - ' . ($W{ct_edit}->FileName() || ''),
      );
   } else {
      $m->entryconfigure('Open', -command => [\&OpenFile]);
      $W{enhex}->focus();
      $W{top}->configure(-title => $title. ' - ' . $LastBinFile);
   }
}


sub ToggleInstrBit {
   my($bit) = @_;

   $cfg{InstrVar}[$bit] = 1 - $cfg{InstrVar}[$bit];

   my $val = 0;
   $val += $cfg{InstrVar}[$_] * (2 ** $_) for ( 0..NICE::INSTR_SIZE-1 );

   $HexEntry{Value} = sprintf('%08X', $val);
}


sub ShowInstructionCode {
   my($val) = @_;

   my $m = init_memory(4);
   my $c = init_code($m);

   $m->[0]->from_Dec($val);

   disassemble($m, $c);

   my @cparts = split(/,?\s+/, $c->[0], -1);

   @cparts = ($cparts[0], '', @cparts[1..$#cparts]);

   $cfg{InstrCode}[$_] = $cparts[$_] ? $cparts[$_] : ''
      for ( 0..$#{$cfg{InstrCode}} );
}


sub ErrMsg {
   my(@str) = @_;
   my($msg) = join('', @str);

   $W{top}->messageBox(
      -icon    => 'error',
      -type    => 'OK',
      -title   => 'ERROR!',
      -message => $msg,
   );
}


sub Assemble {

   $W{ro_status}->delete('1.0', 'end');

   my $err = [];
   my $bin = [];

   my $lines = [
      split(/\n/, $W{ct_edit}->Subwidget('scrolled')->Contents())
   ];

   unless ( assemble($lines, $bin, $err) ) {
      $W{ro_status}->insert(
         'end',
         sprintf("%-8s: %s on line %s\n", $_->{level}, $_->{text}, $_->{line}),
      ) for @$err;
      return;
   }

   $W{ro_status}->insert('end', 'assemble OK');

   my $fn = $W{ct_edit}->FileName();
   return unless $fn;

   my($name, $dir) = fileparse($fn, qr{\..*?});

   my $outfn = catfile($dir, $name . '.bin');
   open(my $out, '>', $outfn) or do {
      ErrMsg("Couldn't write to $outfn, $!");
      return;
   };

   print $out "$_\n" for @$bin;
   close($out);

   return($outfn);
}


sub PLabFrame {
   my($w, $name) = @_;

   my $f = $w->LabFrame(
      -labelside   => 'acrosstop',
      -label       => $name,
      -relief      => 'sunken',
   );

   $f->Subwidget('label')->configure(
      -pady        => 0,
      -borderwidth => 0,
   );
   $f->Subwidget('label')->form(
      -pady => 0,
   );

   my $y = $f->Subwidget('label')->reqheight / 2;

   $f->Subwidget('border')->form(-pady => 0, -top => $y);

   $f->Subwidget('frame')->configure(
      -borderwidth => 0,
   );

   my $ph = int($y - ($f->Subwidget('border')->cget(-bd)));
   $ph = 0 if $ph < 0;

   $f->Subwidget('pad')->form(-bottom => $ph);

   return($f);
}


sub ModEditorBindings {
   my($ed, $bind) = @_;

   $ed->eventDelete('<<Redo>>', '<Control-y>');
   $ed->eventAdd('<<Redo>>', '<Control-z>');
   $ed->bind(ref($ed), '<Key-F2>', '');

   my $inscmd = $ed->bind(ref($ed), '<Key-Insert>');
   $ed->bind(
      ref($ed),
      '<Key-Insert>',
      sub {
         if ( $ed->OverstrikeMode() ) {
            $ed->configure(-insertwidth => 1);
         } else {
            $ed->configure(-insertwidth => 3);
         }
         $inscmd->Call($ed);
      },
   );


   foreach my $cmd ( keys(%$bind) ) {

      my $del_char = $bind->{$cmd} =~ s/-$//;

      if ( my($sub) = $cmd =~ /^&(\S+)/ ) {
         $ed->bind($bind->{$cmd}, [$ed_functions{$sub}, $ed]);
         next;
      }

      my @mcmd = split(/\//, $cmd);
      $_ =~ s/_/ /g for @mcmd;

      $ed->bind(
         $bind->{$cmd},
         sub {
            $ed->delete('insert-1char', 'insert') if $del_char;
            $ed->menu->entrycget(
               $mcmd[0], '-menu',
            )->invoke($mcmd[1]);
         }
      );
   }
}


sub CreateOutput {

   $W{Output} = $W{top}->Toplevel(
      -title => 'Output',
   );
   $W{Output}->bind(
      '<Destroy>',
      sub {
         delete($W{Output});
         delete($W{OutputTags});
      },
   );
   $W{Output}->iconimage($icon);

   $W{roOutput} = $W{Output}->ROText(
      -width      => $cfg{OutputX},
      -height     => $cfg{OutputY},
      -font       => $cfg{OutputFont},
      -background => '#000000',
      -foreground => '#FFFFFF',
      -wrap       => 'char',
   )->pack(qw/-expand yes -fill both/);

   my $size = $cfg{OutputX} * $cfg{OutputY};
   my $end  = $cfg{OutputWin} + $size - 1;
   UpdateOutput($_) for $cfg{OutputWin}..$end;

   $W{Output}->update;

   my $w = $W{Output}->width();
   my $h = $W{Output}->height();
   my($mx, $my) = $W{Output}->maxsize();
   my $top  = $my - $h;
   my $left = $mx - $w;
   my $geom = "${w}x$h+$left+$top";
   $W{Output}->geometry($geom);
}


sub UpdateOutput {
   my($addr) = @_;

   CreateOutput() unless ref($W{Output}) eq 'Tk::Toplevel';

   my(@bg, @fg);
   (@bg[0..2], @fg[0..2]) =
      (split(//, "$Memory->[$addr]"))[0..5];

   @bg = map { $_ x 2 } @bg;
   @fg = map { sprintf("%x", (0xF - hex($_)) ) x 2 } @fg;

   my $bg = '#' . join('', @bg);
   my $fg = '#' . join('', @fg);

   my $ord  = $Memory->[$addr]->Chunk_Read(8, 0);
   my $char = ($ord > 0x1F and $ord < 0x80) ?
              chr($ord)                     :
              ( $ord == 0 ? ' ' : "\x1F" )  ;

   my $y = $addr - $cfg{OutputWin};

   $W{OutputTags} = {} unless exists($W{OutputTags});

   my $tag = $bg.$fg;
   unless ( exists($W{OutputTags}{$tag}) ) {
      $W{roOutput}->tagConfigure(
         $tag,
         -foreground => $fg,
         -background => $bg,
      );
      $W{OutputTags}{$tag} = 1;
   }

   #print "setting >>$char<< at 1,$y with tag $tag\n";

   $W{roOutput}->delete("1.$y");
   $W{roOutput}->insert("1.$y", $char);

   $W{roOutput}->tagAdd($tag, "1.$y")

}

__DATA__
/* XPM */
static char *icon[] = {
/* width height ncolors chars_per_pixel */
"32 32 9 1",
/* colors */
"  c #D0D0D0",
"X c #404040",
". c #606060",
"> c #707070",
"# c #808080",
"@ c #909090",
"+ c #A0A0A0",
": c #B0B0B0",
"; c #C0C0C0",
/* pixels */
"                                ",
"                                ",
"                                ",
"                                ",
"                                ",
"                                ",
"                    ..          ",
"                  ..#>>         ",
"                ..#>>>>>        ",
"              ..##>#>>>>>       ",
"            ..###>#>#>>>..      ",
"          ..#@@###>##>>>>>.     ",
"        ..+@@@#@####>#>>>>..    ",
"      ..:#+@+@@@#@###>#>>>>..   ",
"    ..;#++++@@@#@#@####>#>>>..  ",
"  ..#;:::++@+@+@#@@###>#>>>.XX  ",
" .:;;;::+:+:@+@@@@#@#>##>.XX X  ",
" ..;;;;::++++@++@#@####.XXX +   ",
" .@.;;;::::++++@@@@##.XX XX .   ",
" .@@...X::+:+++@@@@.XXX + X .   ",
" X@@.X;;.::+++@+#.XX XX .   .   ",
"   @X.@.@;::++#.XXX + X .   .   ",
"    .X@.#;:::..X .. .   X       ",
"      @.;;:.X.. + . X   .       ",
"       .@;.. .. .   .           ",
"       X@@. + . .   .           ",
"        @@. .   X               ",
"         @. X   .               ",
"            .                   ",
"            .                   ",
"                                ",
"                                "
};
